
/*
 * nuuo_uboot.c
 *
 * LEDs connected to P1.0 and P1.6, UART on pins P1.1 and P1.2
 *
 * build command: msp430-gcc -Wall -mmcu=msp430g2553 nuuo_uboot.c -o nuuo -> make nuuo
 * target: MSP430G2553
 * Author: Xavier Bourgeois - xavier@monorailc.at
 * License: Creative Commons - BY-NC
 * Article:
 * Date: 2025-10-11 - 2025-10-20
 */

#define MAX_LEN	40
#define DELAY_FSM	10	//ms

//#define DEBUG

//#include <msp430g2553.h>
#include <msp430.h>
#include <string.h>
#include <stdio.h>

char linbuf[MAX_LEN] = {0};
const char str1[] = "Hit Ctrl + C to stop autoboot:\0"; // '\n' needed for terminal debug
const char str2[] = "=>\0";
unsigned long milliseconds = 0;
unsigned long seconds = 0;

uint8_t state = 0;

void toggle_LED(void) {
	P1OUT ^= 0x01;				//XOR P1.0 to toggle it
}

void toggle_LED2(void) {
	P1OUT ^= 1 << 6;			//XOR P1.6 to toggle it
}

int puts(const char *str) {
	static uint8_t i, len;
	len = strlen(str);
	//len = MAX_LEN;

	for(i=0; i<len; i++) {
		while(!(UC0IFG & UCA0TXIFG));
		UCA0TXBUF = (unsigned char) str[i];
	}
	return len;
}

int putchar(int c) {
	while(!(UC0IFG & UCA0TXIFG));
	UCA0TXBUF = c;
	return 0;
}

void addchar(char c) {
	static uint8_t i;
	linbuf[MAX_LEN-1] = c;
	for(i = 1; i < MAX_LEN; i++) {
		linbuf[i-1] = linbuf[i];
	}
	//putchar(c);
}

void clearbuf() {
	static uint8_t i;
	for(i = 1; i < MAX_LEN; i++) {
		linbuf[i] = '\0';
	}
}

void listening_cond(const char *const_str) {
	static char *match = NULL;

	match = strstr(linbuf, const_str);

	#ifdef DEBUG
		if(match != NULL)
			puts(linbuf);
	#endif
	if(match != NULL) {
		if(state == 0)			// 1st match done, ready for 2nd match
			state = 1;
		if(state == 2)			// 2nd match done
			state = 3;
		match = NULL;
		//resetbuf();
		#ifdef DEBUG
			putchar(state + '0'); // 0-3 number
			putchar('\n');
		#endif
	}
}

void fsm() {
	/*
	 * 0: default, listening for 1st string
	 * 1: 1st string matched, waiting 100ms, sending '^C'
	 * 2: listening for 2nd string
	 * 3: 2nd string matched, waiting 100ms, sending "boot\n"
	 */
	static unsigned long delay = 0;
	switch(state) {
		case 1:
			if(delay == 0)
				delay = milliseconds;
			if(milliseconds - delay >= 100) {
				putchar(0x03);
				//puts("^C\n");
				delay = 0;
				state = 2;
				//clearbuf();
				toggle_LED();
			}
			break;
		case 2:
			listening_cond(str2);
			break;
		case 3:
			if(delay == 0)
				delay = milliseconds;
			if(milliseconds - delay >= 100) {
				puts("boot\n");
				delay = 0;
				state = 0;
				clearbuf();
				toggle_LED2();
			}
			break;
		default:
			listening_cond(str1);
			break;
	}
}

__attribute__ ((interrupt(USCIAB0RX_VECTOR))) void uart_RX_ISR(void) {
	addchar(UCA0RXBUF);
	IE2 |= UCA0TXIE;			// Enable USCI_A0 TX interrupt
}

__attribute__ ((interrupt(TIMER0_A0_VECTOR))) void timer_A_ISR(void) {
	//fsm();					// misses characters at 1kHz, run slower instead
	milliseconds++;
	if(!(milliseconds % 1000)) {
		seconds++;
	}
	if(!(milliseconds % DELAY_FSM)) {
		fsm();					// 100 Hz
	}
}

void init(void) {
	WDTCTL = WDTPW + WDTHOLD;	// Disable WDT
	P1DIR = 0x00;				// Reset P1
	P1OUT = 0x00;

	BCSCTL1 = CALBC1_16MHZ;
	DCOCTL = CALDCO_16MHZ;		// All P1.x reset
	P1SEL = BIT1 + BIT2 ;		// P1.1 = RXD, P1.2=TXD
	P1SEL2 = BIT1 + BIT2 ;		// P1.1 = RXD, P1.2=TXD

	UCA0CTL1 |= UCSSEL_2;		// CLK = SMCLK

	//SLAU144K Table 15-4
	/*
	UCA0BR0 = 0x08;				// 1MHz - 115200 => 0x08-0x00
	UCA0BR1 = 0x00;				//
	UCA0MCTL = 0x06 << 1;		// Modulation UCBRSx = 6
	*/
	UCA0BR0 = 0x8A;				// 16MHz - 115200 => 0x8A-0x00
	UCA0BR1 = 0x00;				//
	UCA0MCTL = (0x07 << 1);		// Modulation UCBRSx = 7

	UCA0CTL1 &= ~UCSWRST;		// **Initialize USCI state machine**
	IE2 |= UCA0RXIE;			// Enable USCI_A0 RX interrupt

	CCR0 = 2000;
	CCTL0 = CCIE;				// CCR0 interrupt enabled
	TACTL = TASSEL_2 + ID_3 + MC_1;// SMCLK, /8 prescaler, up-mode
	//TACTL |= TACTL + TAIE;		// enable interrupt

	P1DIR |= 0x01 + (1 << 6);		// P1.0 + P1.6 outputs

	__enable_interrupt();
	__bis_SR_register(LPM0_bits + GIE);	// sleep mode + enable interrupts until ISR
}

int main(void) {

	init();
	while(1);
	return 0;
}
